<?php
/**
 * Chaty Analytics Pro Feature
 *
 * @author  : Premio <contact@premio.io>
 * @license : GPL2
 * */

if (defined('ABSPATH') === false) {
    exit;
}
?>

<div class="container">
    <header class="flex py-4 flex-col items-start sm:flex-row sm:justify-between">
        <a href="<?php echo esc_url( $this->getDashboardUrl() ) ?>">
            <img src="<?php echo esc_url(CHT_PLUGIN_URL.'admin/assets/images/logo-color.svg'); ?>" alt="Chaty" class="logo">    
        </a>
    </header>

    <div class="bg-white flex rounded-lg border border-solid border-[#EAEFF2]">
        <div class="px-7 py-8 flex-1">
            <h2 class="text-cht-gray-150 font-medium font-primary text-xl md:text-3xl max-w-[400px] md:leading-[40px]"><?php esc_html_e("Unlock Chaty Analytics. What can you use it for?", "chaty") ?></h2>
            <ul class="mt-7 flex flex-col space-y-2">
                <li class="flex items-center py-6 px-7 bg-[#F9FAFB] rounded-md space-x-6 text-cht-gray-150 text-lg font-primary">
                    <img width="42" height="59" src="data:image/png;base64,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" alt="">
                    <span class="max-w-[305px]"><?php esc_html_e("Discover which chat channels are most commonly used", "chaty") ?></span>
                </li>
                <li class="flex items-center py-6 px-7 bg-[#F9FAFB] rounded-md space-x-6 text-cht-gray-150 text-lg font-primary">
                    <img width="42" height="59" src="data:image/png;base64,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" alt="">
                    <span><?php esc_html_e("Get full stats on your widgets and turn data into actionable steps to increase conversation rate", "chaty") ?></span>
                </li>
                <li class="flex items-center py-6 px-7 bg-[#F9FAFB] rounded-md space-x-6 text-cht-gray-150 text-lg font-primary">
                    <img width="42" height="59" src="data:image/png;base64,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" alt="">
                    <span><?php esc_html_e("Unlock your widgets' open-rate and find out how your call-to-action messages are performing", "chaty") ?></span>
                </li>
            </ul>

            <div class="flex items-center mt-5 space-x-3">
                <a class="btn rounded-lg drop-shadow-3xl font-normal" href="<?php echo esc_url($this->getUpgradeMenuItemUrl()); ?>">
                    <?php esc_html_e('Upgrade To Pro', 'chaty'); ?>
                </a>
                <a class="new-demo-button btn rounded-lg font-normal bg-gray-100 border-gray-400 text-cht-gray-150 hover:bg-cht-gray-150 hover:text-white" href="javascript:;" class="new-demo-button"><?php esc_html_e("Demo", "chaty") ?></a>
            </div>
        </div>
        <div class="hidden xl:inline-block w-[520px] mt-5">
            <img class="h-full w-auto" src="<?php echo esc_url(plugins_url('../../admin/assets/images/license-image.png', __FILE__)); ?>" alt="Chaty" class="logo">
        </div>
    </div>
</div>

<div class="analytics-popup">
    <div class="analytics-overlay"></div>
    <div class="analytics-popup-content">
        <div class="popup-data">
            <div class="close-popup">
                <a href="javascript:;"><span class="dashicons dashicons-no-alt"></span></a>
            </div>
            <img src="<?php echo CHT_PLUGIN_URL ?>/admin/assets/images/analytics-image.png" />
        </div>
    </div>
</div>
<script>
    jQuery(document).ready(function(){
        jQuery(document).on("click", ".analytics-popup-content .close-popup a, .analytics-overlay, .analytics-popup-content", function(e){
            jQuery(".analytics-popup").hide();
        });
        jQuery(document).on("click", ".new-demo-button", function(e){
            jQuery(".analytics-popup").show();
        });
        jQuery(document).on("click", ".analytics-popup-content img", function(e){
            e.stopPropagation();
        });
    });
</script>
