<?php

/**
 * @package Auto-Install Free SSL
 * This package is a WordPress Plugin. It issues and installs free SSL certificates in cPanel shared hosting with complete automation.
 *
 * @author Free SSL Dot Tech <support@freessl.tech>
 * @copyright  Copyright (C) 2019-2020, Anindya Sundar Mandal
 * @license    http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License, version 3
 * @link       https://freessl.tech
 * @since      Class available since Release 1.0.0
 *
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 */
namespace AutoInstallFreeSSL\FreeSSLAuto;

use  DateTime ;
use  AutoInstallFreeSSL\FreeSSLAuto\Acme\Factory ;
use  AutoInstallFreeSSL\FreeSSLAuto\Admin\HomeOptions ;
use  Exception ;
//Common actions, even if the control panel is not cPanel
class Controller
{
    /**
     * Initiates the Controller class.
     */
    public function __construct()
    {
        if ( !defined( 'ABSPATH' ) ) {
            die( __( "Access Denied", 'auto-install-free-ssl' ) );
        }
        $this->logger = new Logger();
    }
    
    /**
     * Make array of the domains pointing to the same document root of a domain.
     *
     * @param array $domains_array_raw
     * @param array $domains_to_exclude
     *
     * @return array
     */
    public function domainsArray( $domains_array_raw, $domains_to_exclude )
    {
        $domains_array = [];
        foreach ( $domains_array_raw as $domain ) {
            
            if ( !\in_array( $domain, $domains_to_exclude, true ) || $this->sslRequiredForFirstTime( $domain ) ) {
                $domains_array[] = $domain;
            } else {
                $this->logger->log( $domain . " is in your exclusion list, skipping it." );
            }
        
        }
        //remove offline domains
        $domains_online = [];
        foreach ( $domains_array as $key => $domain ) {
            
            if ( false === strpos( $domain, '*.' ) ) {
                //check if domain is online, only for non-wildcard domains
                $socket = @fsockopen(
                    $domain,
                    80,
                    $errno,
                    $errstr,
                    30
                );
                
                if ( $socket ) {
                    //Domain is online
                    
                    if ( strpos( $domain, 'mail.' ) !== false && strpos( $domain, 'mail.' ) === 0 ) {
                        $dns = dns_get_record( $domain, DNS_A );
                        
                        if ( $dns[0]['type'] == "A" && $dns[0]['ip'] == $_SERVER['SERVER_ADDR'] ) {
                            $domains_online[] = $domain;
                        } else {
                            $this->logger->log( "This server's IP is " . $_SERVER['SERVER_ADDR'] . ". But " . $domain . " points to different IP (" . $dns[0]['ip'] . "). So, skipping it." );
                        }
                    
                    } else {
                        $domains_online[] = $domain;
                    }
                
                } else {
                    //domain offline
                    $this->logger->log( $domain . ' is offline. Skipping it.' );
                }
            
            } else {
                //Domain is wildcard and will be validated by DNS-01 challange, so, online check not required
                $domains_online[] = $domain;
            }
        
        }
        /**
         * Rearrange array if the website's domain is in $domains_online
         */
        $this_domain = aifs_get_domain( true );
        
        if ( in_array( $this_domain, $domains_online ) ) {
            $temp_domains_array = $domains_online;
            unset( $domains_online );
            $group_1 = [];
            $group_2 = [];
            foreach ( $temp_domains_array as $domain ) {
                
                if ( $this_domain == $domain ) {
                    $group_1[] = $this_domain;
                } elseif ( "www." . $this_domain == $domain ) {
                    $group_1[] = "www." . $this_domain;
                } else {
                    $group_2[] = $domain;
                }
            
            }
            $domains_online = array_merge( $group_1, $group_2 );
        }
        
        return $domains_online;
    }
    
    /**
     * Make array of the domains pointing to the same document root of a domain.
     * Without any filter (as is/raw)
     *
     * @param array $single_domain
     *
     * @return array
     * @since 3.2.10
     */
    public function domains_array_raw( $single_domain )
    {
        $domains_array = [];
        $domains_array[] = $single_domain['domain'];
        
        if ( \strlen( $single_domain['serveralias'] ) > 1 ) {
            $domains = explode( ' ', $single_domain['serveralias'] );
            foreach ( $domains as $domain ) {
                if ( $single_domain['domain'] !== $domain ) {
                    $domains_array[] = $domain;
                }
            }
        }
        
        return $domains_array;
    }
    
    /**
     * Checks whether SSL certificate MUST required for the given domain
     * @param $domain
     *
     * @return bool
     */
    public function sslRequiredForFirstTime( $domain )
    {
        //For free version always return true
        return true;
    }
    
    /**
     *
     *
     * Get the wildcard base domain.
     *
     * @param string $domain_as_is
     *
     * @return string
     */
    public function getWildcardBase( $domain_as_is )
    {
        $registeredDomain = getRegisteredDomain( $domain_as_is );
        if ( null === $registeredDomain ) {
            return false;
        }
        //compute wildcard domain
        
        if ( \strlen( $domain_as_is ) > \strlen( $registeredDomain ) ) {
            //may be it's a subdomain
            $part = str_replace( $registeredDomain, '', $domain_as_is );
            $domain_elements_array = explode( '.', $part );
            if ( 2 === \count( $domain_elements_array ) && 'www' === $domain_elements_array[0] ) {
                //www.domain.com is not considered as wildcard domain
                return $domain_as_is;
            }
            //get the position of first . and ONLY replace the part left to it with *
            $pos = strpos( $domain_as_is, '.' );
            return '*' . substr( $domain_as_is, $pos );
        } elseif ( \strlen( $domain_as_is ) === \strlen( $registeredDomain ) ) {
            return $domain_as_is;
        }
    
    }

}